----------------------------------------------------------------------------------
-- Company:
-- Engineer:       Gerald Ebert
--
-- Create Date:    20:06:31 05/02/2007
-- Design Name:
-- Module Name:    BOOTRAM68k - Behavioral
-- Project Name:   BOOTRAM68k
-- Target Devices: XC9536-xxPC44
-- Tool versions:
-- Description:
--
-- Dependencies:
--
-- Revision:       1.20
-- Revision 0.01 - File Created
-- Additional Comments:
--
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;


---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
library UNISIM;
use UNISIM.VComponents.all;

ENTITY BOOTRAM68Main IS
  PORT ( busdata6_7    : IN  STD_LOGIC_VECTOR (1 DOWNTO 0);
         busadr0_7     : IN  STD_LOGIC_VECTOR (7 DOWNTO 0);
         busadr15_19   : IN  STD_LOGIC_VECTOR (4 DOWNTO 0);
         busiorq0      : IN  STD_LOGIC;
         busmreq0      : IN  STD_LOGIC;
         buswr0        : IN  STD_LOGIC;
         busreset0     : IN  STD_LOGIC;
         busbanken     : IN  STD_LOGIC;          -- BANKEN from NDR/ECB-Bus

         banken_en0    : IN  STD_LOGIC;          -- Bootoption if jumper is set
         ramsel_switch : IN  STD_LOGIC_VECTOR (3 DOWNTO 0);
         ramsel_mode   : IN  STD_LOGIC;          -- Lower Memory if jumper is set

         banken_out    : OUT  STD_LOGIC;         -- Simulates open collector gate
         banken_led    : OUT  STD_LOGIC;         -- Drives Boot Indicator LED
         ram0_cs0      : OUT  STD_LOGIC;         -- Chip Select RAM0
         ram1_cs0      : OUT  STD_LOGIC;         -- Chip Select RAM1
         booteprom_cs0 : OUT  STD_LOGIC;         -- Chip Select BootEPROM
         gpeprom_cs0   : OUT  STD_LOGIC;         -- Chip Select GP-EPROM
			databus_en0   : OUT  STD_LOGIC);	       -- Enables '645 bus driver

END BOOTRAM68Main;

ARCHITECTURE Behavioral OF BOOTRAM68Main IS

  CONSTANT lowsignal   : STD_LOGIC := '0';
  CONSTANT adrbankboot : STD_LOGIC_VECTOR (7 DOWNTO 0) := "11001000";
  CONSTANT adrgpeprom  : STD_LOGIC_VECTOR (3 DOWNTO 0) := "1000";

  SIGNAL   bankreg : STD_LOGIC_VECTOR (1 DOWNTO 0) := "00";  -- Registerbits
  SIGNAL   regmatch, regclk, komp_ram, komp_adrgp, ramsel, gpepromsel : STD_LOGIC;
  SIGNAL   iram_cs0, ibooteprom_cs0, igpeprom_cs0 : STD_LOGIC;
  SIGNAL   ibanken_out : STD_LOGIC;

BEGIN
  -- Calc enable signal for registers
  regmatch <= '0' WHEN busadr0_7 = adrbankboot ELSE '1';

  bankregclk : PROCESS ( regmatch, busiorq0, buswr0 )
    BEGIN -- FlipFlop to create a clean regclk
     IF ( regmatch = '1' OR busiorq0 = '1' ) THEN
	    regclk <= '1';
	  ELSIF ( buswr0 = '0' AND buswr0'event ) THEN
       regclk <= '0';
	  END IF;
  END PROCESS bankregclk;

-- Handle BankBoot registers (d6/d7)
  bankregister : PROCESS ( busreset0, regclk, busdata6_7 )
  BEGIN
    IF ( busreset0 = '0' ) THEN
      bankreg <= "00";   -- Default states like original BANKBOOT
    ELSIF ( regclk = '1' AND regclk'event ) THEN
      bankreg <= busdata6_7;  -- Enable GP-EPROM if '1'; Disable BootEPROM if '1'
    END IF;
  END PROCESS bankregister;

  -- Compare bankaddresses for work modes
  komp_ram   <= '1' WHEN busadr15_19(4 DOWNTO 1) < ramsel_switch ELSE '0';
  komp_adrgp <= '1' WHEN busadr15_19(4 DOWNTO 1) = adrgpeprom ELSE '0';
  -- Get memory selection
  ramsel     <= komp_ram WHEN ramsel_mode = '0' ELSE (NOT komp_ram);
  gpepromsel <= bankreg(0) AND komp_adrgp;
  -- Calc which memory device to be enabled
  iram_cs0       <= busmreq0 OR (NOT (busbanken AND ramsel)) OR gpepromsel;
  ibooteprom_cs0 <= busmreq0 OR busbanken OR bankreg(1);
  igpeprom_cs0   <= busmreq0 OR (NOT (busbanken AND gpepromsel));
  -- Transport signals
  databus_en0   <= iram_cs0 AND ibooteprom_cs0 AND igpeprom_cs0;
  ram0_cs0      <= iram_cs0 OR busadr15_19(4);
  ram1_cs0      <= iram_cs0 OR (NOT busadr15_19(4));
  booteprom_cs0 <= ibooteprom_cs0;
  gpeprom_cs0   <= igpeprom_cs0;

  -- Handle bankboot logic and transfer signals
  ibanken_out <= NOT(bankreg(1) OR banken_en0) WHEN busadr15_19(0) = '0'
                                               ELSE '0';           -- high active
  banken_led  <= NOT bankreg(1);                                   -- high active
  -- Simulate open collector gate to pull down BANKEN signal
  bankendrv : OBUFE PORT MAP ( i => lowsignal, o => banken_out, e => ibanken_out);

END Behavioral;

