/* CBASE ist ein freies, eigenständig entwickeltes Datenbanksystem für den NKC 68000,
inspiriert von den klassischen dBase-Systemen der 1980er Jahre (dBase II / III).

Ziel des Projekts ist es, die Funktionsweise früher datenbankbasierter Anwendungen auf Retro-Hardware nachvollziehbar, erlernbar und erweiterbar zu machen – nicht die exakte Nachbildung oder Emulation eines kommerziellen Produkts.


Abgrenzung:

CBASE ist keine Portierung, Emulation oder Rekonstruktion von dBase II, dBase III oder verwandter kommerzieller Software.

    Es wurde kein Original-Quellcode verwendet
    Es wurden keine Binärformate oder Routinen übernommen
    Die Entwicklung erfolgte ausschließlich auf Basis:
    öffentlich dokumentierter Dateiformate (DBF)
    historischer Literatur
    eigener Analyse
    praktischer Erfahrung und Erinnerung an frühere Systeme

CBASE ist eine eigenständige Neuentwicklung, die sich konzeptionell an der Arbeitsweise historischer Datenbanksysteme orientiert.


Lernsystem & Community-Projekt

CBASE wurde als offenes Lern- und Experimentiersystem entwickelt.
Der vollständige Quelltext wird gemeinsam mit dem ausführbaren Programm bereitgestellt, um:

den Aufbau klassischer Datenbanksysteme zu verstehen
eigene Erweiterungen zu entwickeln
Funktionen zu verändern oder wiederzuverwenden
Retro-Softwareentwicklung auf dem NKC 68000 zu fördern
Das Projekt richtet sich ausdrücklich an Entwickler, Bastler und Retro-Computer-Interessierte.


Weiterverwendung des Codes

Die Nutzung, Modifikation und Weiterverwendung des Quelltexts im Rahmen nicht-kommerzieller Projekte ist ausdrücklich erwünscht.

Ziel ist es, Wissen zu teilen und den Austausch innerhalb der Retro-Community zu fördern – insbesondere im Umfeld des NKC 68000.


Historischer Kontext

Datenbanksysteme wie dBase II und dBase III prägten in den 1980er Jahren die Anwendungsentwicklung auf Mikrocomputern maßgeblich.

CBASE greift diese Ideen auf und überträgt sie auf ein modernes Retro-System – den NKC 68000 – ohne den Anspruch auf Vollständigkeit oder historische Exaktheit im Detail.

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "cbase.h"

#include "util.h"
#include "db.h"
#include "sort.h"
#include "locate.h"
#include "export.h"
#include "create.h"
#include "record.h"



#include "help.h"
#include "messages.h"

#include "show.h"



// Datenbank und Datenbank geöffnet Flag
static db_t db;
static int db_opened = 0;

// LOCATE und CONT, STRUCT in DB.h
static locate_ctx_t locate_ctx;


static void cmd_open(char *arg)
{
    if (db_opened && db.dirty) {
        int ch = ask_yes_no(MSG_DB_DIRTY_WARNING);
        printf("\r\n");
        if (ch != DLG_YES)
            return;
    }


    // Leerzeichen am Argumentenanfang entfernen
    while (*arg == ' ')
        arg++;
    
    // Ist nächstes Argument DIR
    if (strcmp(arg, "DIR") == 0) {
        util_dir_drive("*.DBF");
        return;
    }

    // Wenn eine Datenbank offen ist, dann schliessen
    if (db_opened) {
        printf("\nClosing %s\r\n", db.name);
        db_close(&db); 
        db_opened = 0;
    }

    // Ohne weiteres Argument beenden, Datenbankname muss eingegeben werden
    if (!arg || !*arg) {
        printf(MSG_DB_USE_DB);
        return;
    }

    // Wenn Datenbank nicht geöffnet werden kann, dann Meldung und zurück
    if (db_open(&db, arg)) {
        printf("\nCannot open %s\n", arg);
        return;
    }

    // LOCATE und CONT deaktivieren
    locate_ctx.active = 0;
    // Flag für geöffnete Datenbank setzen und Meldung
    db_opened = 1;
    printf("\nDatabase %s opened\r\n", db.name);

}


/* geöffnete Datenbak speichern (inkl. gelöscht markierter Datensätze)*/
static void cmd_save(void)
{
    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }


    if (db_save(&db) == 0) {
        iprintf(MSG_DB_SAVED);
        db.dirty = 0;
    }
    else
        iprintf(MSG_DB_NOT_SAVED);

}


/// CREATE Datenbank
void cmd_create(char *arg)
{

    if (db_opened && db.dirty) {
        int ch = ask_yes_no(MSG_DB_DIRTY_WARNING);
        printf("\r\n");
        if (ch != DLG_YES)
            return;
    }

    char fname[13];
    create_ctx_t ctx;

    memset(&ctx, 0, sizeof(ctx));
    ctx.record_len = 1;   /* Delete-Flag */

    /* Dateiname prüfen */
    while (*arg == ' ')
        arg++;

    if (!is_valid_db_name(arg)) {
        printf(MSG_DB_NOT_VALID);
        return;
    }

    make_db_filename(fname, sizeof(fname), arg);

    printf("Creating %s\r\n", fname);
    // Einleitender Text für CREATE
    printf("\nEnter field definitions:\r\n");
    printf("FORMAT: NAME,TYPE,LENGTH,DECIMAL\r\n");
    printf("Example: PREIS,N,6,2\r\n");
    printf("RETURN on empty line = finish\r\n");
    printf("ESC = cancel CREATE\r\n\r\n");



    while (1) {

        char line[128];
        int rc;

        printf("Field %03d -> ", ctx.field_count + 1);
        rc = read_line_ci(line, sizeof(line));

        /* ESC */
        if (rc < 0) {
            printf(MSG_CREATE_CANCEL);
            return;
        }

        /* Leere Eingabe = fertig */
        if (rc == 0)
            break;


        if (ctx.field_count >= MAX_CREATE_FIELDS) {
            printf(MSG_CREATE_TOO_MANY_FIELDS);
            continue;
        }

        if (!parse_create_field(line, &ctx)) {
            printf(MSG_CREATE_INVALID_FIELD);
            printf("Please re-enter field %03d\r\n", ctx.field_count + 1);
            continue;
        }

        printf(MSG_CREATE_VALID_FIELD);
    }

    // Definiton ind Database speichern

    /* DBF-Dateiname bilden */
    //make_db_filename(fname, sizeof(fname), dbname);

    if (create_save(fname, ctx.fields, ctx.field_count, ctx.record_len) == 0) {
        printf(MSG_CREATE_DB_CREATED);
        cmd_open(arg);   /* automatisch öffnen */
    } else {
        printf(MSG_CREATE_DB_NOT_CREATED);
    }


}




// PACK
// PACK entfernt als gelöscht markierte Datensätze
static void cmd_pack(void)
{
    int rc;
    uint32_t removed;

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    // Sicherheitsabfrage, da PACK auch speichert
    rc = ask_yes_no(MSG_PACK_QUESTION);

    if (rc != DLG_YES) {
        // Pack abgebrochen
        printf(MSG_PACK_ABORTED);
        return;
    }

    removed = db_pack(&db);

    if (removed == 0) {
        // Keine gelöschten Datensätze enthalten
        printf(MSG_PACK_NO_PACK);
        return;
    }

    db_save(&db);

    /* Zustände zurücksetzen */
    db.current_rec = 0;
    db.eof = 0;
    locate_ctx.active = 0;


    printf("%lu records removed.\r\n",
           (unsigned long)removed);
}


// Datenbank unter neuem Namen speichern

static void cmd_copy(char *arg)
{
    char fname[13];
    jdfcb_t testfcb;
    int exists = 0;
    int ch;
    int copy_loc = 0;   /* NEU */

    if (!db_opened) {
        iprintf(MSG_NO_DB);
        return;
    }

    while (*arg == ' ')
        arg++;

    /* COPY DIR */
    if (strcmp(arg, "DIR") == 0) {
        util_dir_drive("*.DBF");
        return;
    }

    /* COPY LOC <name> */
    if (!strncasecmp(arg, "LOC", 3)) {

        if (!locate_ctx.active) {
            iprintf(MSG_SHOW_NO_LOCATE);
            return;
        }

        arg += 3;
        while (*arg == ' ')
            arg++;

        copy_loc = 1;
    }

    /* jetzt MUSS ein Dateiname kommen */
    if (!is_valid_db_name(arg)) {
        iprintf(MSG_DB_NOT_VALID);
        return;
    }

    make_db_filename(fname, sizeof(fname), arg);

    jd_fillfcb(&testfcb, fname);
    if (jd_open(&testfcb) == 0) {
        jd_close(&testfcb);
        exists = 1;
    }

    if (exists) {
        ch = ask_yes_no(MSG_DB_COPY_EXISTS);
        printf("\r\n");

        if (ch != DLG_YES) {
            iprintf(MSG_DB_COPY_CANCEL);
            return;
        }
    }
    if (copy_loc) {

        if (db_copy_loc(&db, fname, &locate_ctx) == 0)
            iprintf(MSG_DB_COPY_LOG_OK);
        else
            iprintf(MSG_DB_COPY_LOC_FAILED);

    } else {

        if (db_copy(&db, fname) == 0)
            iprintf(MSG_DB_COPIED);
        else
            iprintf(MSG_DB_NOT_COPIED);
    }
}




// Zeige Datensätze
// SHOW zeige alle mit Paging
// SHOW 10 zeige alle ab 10. Datensatz
// SHOW 10 5 zeige 5 Datensätze nach dem 10. Date


static void cmd_close(void)
{
 
    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    if (db_opened && db.dirty) {
        int ch = ask_yes_no(MSG_DB_DIRTY_WARNING);
        printf("\r\n");
        if (ch != DLG_YES)
            return;
    }

    
    printf("\nClosing %s\r\n", db.name);

    db_close(&db);
    // Flag setzen für geschlossene Datenbank
    db_opened = 0;
    // LOCATE deaktivieren
    locate_ctx.active = 0;

    printf(MSG_DB_OPEN_OK);


}

// APPEND Neue Datensatz am Ende anhängen
static void cmd_append(void)
{
    int rc;

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    while (1) {

        rc = db_append_record(&db);

        if (rc == APPEND_DONE) {

            db.dirty = 1;   /* <<< WICHTIG */

            if (ask_yes_no(MSG_APPEND_ADD_ANOTHER) == DLG_YES) {
                printf("\r\n");
                continue;
            }
            break;
        }

        if (rc == APPEND_SKIP) {
            /* Record nicht gespeichert → APPEND beenden */
            break;
        }

        if (rc == APPEND_ABORT) {
            /* ESC → APPEND sofort beenden */
            break;
        }
    }
    SetCursorxy(1, 23);
    printf("\r\n");
}


// INSERT

static void cmd_insert(char *arg)
{
    uint32_t recno;
    int before = 0;   // 0 = AFTER, 1 = BEFORE
    int blank = 0;

    if (!db_opened) {
        iprintf(MSG_NO_DB);
        return;
    }

    while (*arg == ' ')
        arg++;

    if (!*arg) {
        iprintf("INSERT requires record number\n");
        return;
    }

    if (!strncmp(arg, "BEFORE", 6)) {
        before = 1;
        arg += 6;
    }
    else if (!strncmp(arg, "AFTER", 5)) {
        before = 0;
        arg += 5;
    }

    while (*arg == ' ')
        arg++;

    if (!strncmp(arg, "BLANK", 5)) {
        blank = 1;
        arg += 5;
    }

    while (*arg == ' ')
        arg++;

    if (!isdigit((unsigned char)*arg)) {
        iprintf("Invalid record number\n");
        return;
    }

    recno = (uint32_t)atoi(arg);

    if (recno < 1 || recno > db.header.num_records) {
        iprintf("Record out of range\n");
        return;
    }

    /* 1-basiert → 0-basiert */
    uint32_t insert_pos = before ? (recno - 1) : recno;

    if (db_insert(&db, insert_pos) == 0) {

        db.current_rec = insert_pos;
        db.eof = 0;

        if (!blank) {
            /* sofortige Datenerfassung */
            db_edit_record(&db, db.current_rec);
        }

        iprintf("Record inserted at %lu\n", db.current_rec + 1);
    }
}

// EDIT
static void cmd_edit(char *arg)
{
    uint32_t recno;
    int rc;

    if (!db_opened) {
        iprintf(MSG_NO_DB);
        return;
    }

    if (db.header.num_records == 0) {
        iprintf("No records\n");
        return;
    }

    while (*arg == ' ')
        arg++;

    if (*arg == 0) {
        /* EDIT ohne Parameter */
        recno = db.current_rec;
    } else {
        recno = (uint32_t)atoi(arg) - 1;
    }

    if (recno >= db.header.num_records) {
        iprintf("Invalid record\n");
        return;
    }

    db.current_rec = recno;

    rc = db_edit_record(&db, recno);

    if (rc == EDIT_ABORT) {
        iprintf("Edit cancelled\n");
    } else {
        // Datensatz gespeichert
        printf("\r\n");
    }
}





// Navigation in Datenbank mit GOTO, um aktuellen Record zu bestimmen
static void cmd_goto(db_t *db, const char *arg)
{

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }


    uint32_t n;
    // Goto ohne Argumente gestartet
    if (!arg || !*arg) {
        printf(MSG_GOTO_REQ_ARG);
        print_current_record(db);
        return;
    }

    while (*arg == ' ') arg++;

    if (!strcasecmp(arg, "TOP")) {
        if (db->header.num_records == 0) {
            printf(MSG_NO_RECORDS);
            return;
        }
        db->current_rec = 0;
        db->eof = 0;
        printf("TOP ");
        print_current_record(db);
        return;
    }

    if (!strcasecmp(arg, "BOTTOM")) {
        if (db->header.num_records == 0) {
            printf(MSG_NO_RECORDS);
            return;
        }
        db->current_rec = db->header.num_records - 1;
        db->eof = 0;
        printf("BOTTOM ");
        print_current_record(db);
        return;
    }

    if (!strcasecmp(arg, "NEXT")) {
        if (db->header.num_records == 0) {
            printf(MSG_NO_RECORDS);
            return;
        }
        if (db->current_rec + 1 >= db->header.num_records) {
            printf(MSG_EOF);
            return;
        }
        db->current_rec++;

        printf("NEXT ");
        print_current_record(db);

        return;
    }

    if (!strcasecmp(arg, "PREV")) {
        if (db->header.num_records == 0) {
            printf(MSG_NO_RECORDS);
            return;
        }
        if (db->current_rec - 1 >= db->header.num_records) {
            printf(MSG_EOF);
            return;
        }
        db->current_rec--;

        printf("PREV ");
        print_current_record(db);
        return;
    }

    n = (uint32_t)atoi(arg);

    /* === HIER IST DER KRITISCHE TEIL === */
    if (n < 1 || n > db->header.num_records) {
        printf(MSG_BAD_RECORD);
        return;
    }

    db->current_rec = n - 1;   /* 1-basiert → 0-basiert */
    db->eof = 0;

    printf("GOTO (record %lu)\n", (unsigned long)n);
}


// Kommando SKIP

void cmd_skip(char *arg)
{
    int step = 1;

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    if (arg && *arg) {
        step = atoi(arg);
        if (step == 0)
            step = 1;
    }

    db_skip(&db, step);
}




// Kommando LOCATE zum Suchen des ersten Vorkommens
static void cmd_locate(char *arg)
{
    char *field;
    char *pattern;
    int field_index;
    int rc;

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    if (!arg || !*arg) {
        printf(MSG_LOCATE_REQ_ARG);
        return;
    }

    while (*arg == ' ')
        arg++;

    field = arg;

    while (*arg && *arg != ' ')
        arg++;

    if (*arg)
        *arg++ = '\0';

    while (*arg == ' ')
        arg++;

    if (!*arg) {
        printf(MSG_LOCATE_MISSING_ARG);
        return;
    }

    pattern = arg;

    field_index = find_field_index(&db, field);
    if (field_index < 0) {
        printf(MSG_LOCATE_UNKNOWN_FIELD);
        printf("%s\n", field);
        return;
    }

    /* Locate-Kontext initialisieren */
    memset(&locate_ctx, 0, sizeof(locate_ctx));
    locate_ctx.active = 1;
    locate_ctx.field_index = field_index;

    if (db.fields[field_index].type == 'C') {

        locate_ctx.type = LOC_TYPE_CHAR;
        locate_ctx.op   = LOC_OP_CONTAINS;

        strncpy(locate_ctx.str_value, pattern,
                sizeof(locate_ctx.str_value) - 1);
        locate_ctx.str_value[sizeof(locate_ctx.str_value) - 1] = '\0';

        util_str_toupper(locate_ctx.str_value);

    }
    else if (db.fields[field_index].type == 'N') {

        const char *p = pattern;

        locate_ctx.type = LOC_TYPE_NUM;
        locate_ctx.op   = parse_num_operator(&p);

        locate_ctx.num_value = atof(p);
    }
    else {
        printf(MSG_LOCATE_FIELD_TYPE);
        locate_ctx.active = 0;
        return;
    }
    rc = db_locate(&db, &locate_ctx, 0);


    if (rc == 0) {
        printf(MSG_LOCATE_FOUND);
        print_current_record(&db);
    } else {
        locate_ctx.active = 0;
        printf(MSG_LOCATE_NOT_FOUND);
    }
}



// CONT
// Zeigt nächsten möglichen Treffer nach LOCATE
static void cmd_cont(void)
{
    int rc;

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    if (!locate_ctx.active) {
        printf(MSG_CONT_NO_LOCATE);
        return;
    }

    rc = db_locate(&db,
                   &locate_ctx,
                   db.current_rec + 1);

    if (rc == 0) {
        printf(MSG_LOCATE_FOUND);
        print_current_record(&db);
    } else {
        db.eof = 1;
        printf(MSG_CONT_EOF);
        /* WICHTIG: locate_ctx.active bleibt UNVERÄNDERT */
    }
}



// Zeige Datensätze
// SHOW zeige alle mit Paging
// SHOW 10 zeige alle ab 10. Datensatz
// SHOW 10 5 zeige 5 Datensätze nach dem 10. Datensatz
// SHOW DEL zeige alle gelöscht markierten datensätze
static void cmd_show(char *arg)
{
    uint32_t start;
    uint32_t count = 0;
    uint32_t num1 = 0, num2 = 0;
    int num_count = 0;

    int show_all = 0;
    db_show_mode_t mode = DB_SHOW_NORMAL;

    // SHOW LOC - LOCATE ERgebnis abbilden
    int show_loc = 0;


    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    /* Default: ab aktueller Markierung */
    start = db.current_rec;

    if (!arg || !*arg) {
        // Kein Argument, nur SHOW
        // Zeigt nur aktuellen Datensatz (Pointer)
        count = 1;
    }
    else
    {
        while (*arg == ' ')
            arg++;

        /* SHOW ALL */
        if (!strncmp(arg, "ALL", 3)) {
            show_all = 1;
            arg += 3;
        }

        /* SHOW LOC */
        if (!strncmp(arg, "LOC", 3)) {
            show_loc = 1;
            arg += 3;
        }

        while (*arg == ' ')
            arg++;

        /* SHOW DEL */
        if (!strncmp(arg, "DEL", 3)) {
            mode = DB_SHOW_DELETED;
            arg += 3;
        }

        while (*arg == ' ')
            arg++;

        /* numerische Argumente einlesen */
        if (*arg && isdigit((unsigned char)*arg)) {
            num1 = (uint32_t)atoi(arg);
            num_count++;


            while (*arg && *arg != ' ')
                arg++;
            while (*arg == ' ')
                arg++;

            if (*arg && isdigit((unsigned char)*arg)) {
                num2 = (uint32_t)atoi(arg);
                num_count++;
            }
        }
    }

    // SHOW LOC - LOCATE Auswertung
    if (show_loc) {
        if (!locate_ctx.active) {
            // Kein aktives LOCATE
            printf(MSG_SHOW_NO_LOCATE);
            return;
        }
        // SHOW LOC ausführen
        printf("\r\n");
        cbase_show_loc(&db, &locate_ctx);;
        return;
    }


    /* Start-Logik */
    if (show_all)
        start = 0;

    if (num_count == 1) {
        /* SHOW count  ODER  SHOW ALL count */
        count = num1;
    }
    else if (num_count == 2) {
        /* SHOW start count */
        if (num1 > 0)
            start = num1 - 1;   /* 1-basiert → 0-basiert */
        count = num2;
    }

    printf("\r\n");
    cbase_show(&db, start, count, mode);
}


/// EXPORT
// Export in CSV Datei
static void cmd_export(char *arg)
{
    char fname[13];
    db_export_mode_t mode = DB_EXPORT_CURRENT;
    
    /* führende Leerzeichen */
    while (*arg == ' ')
        arg++;

    // Ist nächstes Argument DIR
    if (strcmp(arg, "DIR") == 0) {
        util_dir_drive("*.CSV");
        return;
    }


    if (!db_opened) {
        iprintf(MSG_NO_DB);
        return;
    }

    /* Argument prüfen */
    if (!arg) {
        iprintf(MSG_EXPORT_ARG);
        return;
    }

    /* führende Leerzeichen */
    while (*arg == ' ')
        arg++;

    /* EXPORT ALL */
    if (!strncmp(arg, "ALL", 3)) {
        mode = DB_EXPORT_ALL;
        arg += 3;
    }
    /* EXPORT LOC */
    else if (!strncmp(arg, "LOC", 3)) {
        mode = DB_EXPORT_LOC;
        arg += 3;
    }

    /* Leerzeichen nach Modifier */
    while (*arg == ' ')
        arg++;

    /* Dateiname vorhanden? */
    if (!*arg) {
        iprintf(MSG_EXPORT_MISSING_FILE);
        return;
    }

    /* LOC ohne aktives LOCATE */
    if (mode == DB_EXPORT_LOC && !locate_ctx.active) {
        iprintf(MSG_EXPORT_NO_LOCATE);
        return;
    }

    /* Dateiname prüfen */
    if (!is_valid_db_name(arg)) {
        iprintf(MSG_DB_NOT_VALID);
        return;
    }

    /* Dateiname bilden */
    make_csv_filename(fname, sizeof(fname), arg);

    /* Aufruf DB-Ebene */
    if (db_export(&db, fname, mode, &locate_ctx) == 0)
        iprintf(MSG_EXPORT_OK);
    else
        iprintf(MSG_EXPORT_NOT_OK);
}



// SORT
static void cmd_sort(char *arg)
{
    char fname[13];
    char *p;
    char *dest;

    sort_key_t keys[SORT_MAX_KEYS];
    int key_count = 0;

    uint32_t *idx = NULL;
    uint32_t count;

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    if (!arg || !*arg) {
        printf(MSG_SORT_ARG);
        return;
    }

    /* führende Leerzeichen */
    while (*arg == ' ')
        arg++;

    /* ON prüfen */
    if (strncmp(arg, "ON", 2) != 0) {
        printf(MSG_SORT_ARG_ON);
        return;
    }

    arg += 2;

    /* -------------------------------------------------
       Sort-Feldliste parsen
       ------------------------------------------------- */
    while (1) {

        int descending = 0;
        int field_index;

        while (*arg == ' ')
            arg++;

        /* TO beendet die Feldliste */
        if (strncmp(arg, "TO", 2) == 0)
            break;

        if (key_count >= SORT_MAX_KEYS) {
            printf("Too many sort fields (max %d)\n", SORT_MAX_KEYS);
            return;
        }

        /* Feldname */
        p = arg;
        while (*arg && *arg != ' ' && *arg != ',')
            arg++;

        if (*arg)
            *arg++ = '\0';

        field_index = find_field_index(&db, p);
        if (field_index < 0) {
            printf("Unknown field: %s\n", p);
            return;
        }

        /* Optional DESC */
        while (*arg == ' ')
            arg++;

        if (strncmp(arg, "DESC", 4) == 0) {
            descending = 1;
            arg += 4;
        }

        keys[key_count].field_index = field_index;
        keys[key_count].descending  = descending;
        key_count++;

        while (*arg == ' ')
            arg++;

        /* weiteres Feld? */
        if (*arg == ',') {
            arg++;
            continue;
        }

        /* sonst weiter mit TO */
    }

    /* -------------------------------------------------
       TO <Datei>
       ------------------------------------------------- */
    if (strncmp(arg, "TO", 2) != 0) {
        printf(MSG_SORT_ARG_TO);
        return;
    }

    arg += 2;
    while (*arg == ' ')
        arg++;

    dest = arg;
    while (*arg && *arg != ' ')
        arg++;

    if (*arg)
        *arg = '\0';

    if (!is_valid_db_name(dest)) {
        printf(MSG_DB_NOT_VALID);
        return;
    }

    make_db_filename(fname, sizeof(fname), dest);

    /* -------------------------------------------------
       Debug / Bestätigung
       ------------------------------------------------- */
    printf("SORT ON ");
    for (int i = 0; i < key_count; i++) {
        printf("%s%s",
               db.fields[keys[i].field_index].name,
               keys[i].descending ? " DESC" : "");
        if (i < key_count - 1)
            printf(", ");
    }
    printf(" TO %s\n", fname);

    /* -------------------------------------------------
       Sort ausführen
       ------------------------------------------------- */
    if (db_sort_prepare(&db, keys, key_count, &idx, &count) != 0) {
        printf(MSG_SORT_FAILED);
        return;
    }

    if (db_sort_to_file(&db, fname, idx, count) == 0)
        printf("SORTED %lu RECORDS TO %s\n",
               (unsigned long)count, fname);
    else
        printf(MSG_SORT_FAILED);

    free(idx);
}



// Datenbank info ausgeben
static void cmd_info(void)
{
    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }
    printf("\r\n");
    db_info(&db);
}


// Einen Datensatz logisch löschen
static void cmd_del(const char *args)
{
    unsigned recno;
    int rc;

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    if (sscanf(args, "%u", &recno) != 1) {
        printf("\nDEL <record>\r\n");
        return;
    }

    rc = db_delete_record(&db, recno);

    if (rc == 0)
        printf("\nRecord %u deleted\r\n", recno);
    else if (rc == 1)
        printf("\nRecord %u already deleted\r\n", recno);
    else
        printf(MSG_BAD_RECORD);
}


// Logisch gelöschten Datensatz wieder aktivieren
static void cmd_undel(char *arg)
{
    uint32_t rec;

    if (!db_opened) {
        printf(MSG_NO_DB);
        return;
    }

    if (!arg) {
        printf(MSG_UNDEL_REQ_NUM);
        return;
    }

    while (*arg == ' ')
        arg++;

    if (!*arg) {
        printf(MSG_UNDEL_REQ_NUM);
        return;
    }

    rec = (uint32_t)atoi(arg);

    if (rec == 0) {
        printf(MSG_BAD_RECORD);
        return;
    }

    rec--;   /* 1-basiert → 0-basiert */

    if (db_undel(&db, rec))
        printf("\nRecord %lu not deleted\n", (unsigned long)(rec + 1));
    else
        printf("\nRecord %lu restored\n", (unsigned long)(rec + 1));
}



static void cmd_help(char *arg)
{
    if (!arg || !*arg) {

        help_general();
        return;
    }

    while (*arg == ' ')
        arg++;

    if (*arg)
        help_command(arg);
    else
        help_general();
}


static void cmd_cls()
{
    // Bildschirm löschen
    gp_clearscreen();
}


static void cmd_dir()
{
    util_dir_drive("*.*");

}
static void cmd_test()
{
test_screen_input();

}


int main(int argc, char **argv)
{
    // BUffer deaktivieren
    setbuf(stdout,NULL);
    setbuf(stdin,NULL);

    char line[80];
    // Bildschirm lösche bei Start
    gp_clearscreen();
    printf("%s %s\r\n", CBASE_NAME, CBASE_VERSION);

   if (argc > 1) {
        if (db_open(&db, argv[1]) == 0) {
            db_opened = 1;
            printf("Using Database %s\r\n", argv[1]);
        } else {
            printf("Cannot open %s\r\n", argv[1]);
        }
    }

    printf(MSG_HELP_FOR);



    for (;;) {

        iprintf(CBASE_PROMPT);

        if (read_line_ci(line, sizeof(line)) < 0)
        {
             printf("\r\n");
             continue;          /* ESC → neue Eingabe */
        }

        if (line[0] == '\0')
            continue;

        if (*line)
            history_add(line);

        /* newline entfernen */
        line[strcspn(line, "\r\n")] = 0;
        util_str_toupper(line);

        if (!strncmp(line, "USE", 3))
            cmd_open(line + 3);
        else if (!strncmp(line, "CREATE", 6))
            cmd_create(line + 6);
        else if (!strcmp(line, "SAVE"))
            cmd_save();
        else if (!strcmp(line, "PACK"))
            cmd_pack();
        else if (!strncmp(line, "COPY", 4))
            cmd_copy(line + 4);
        else if (!strcmp(line, "CLOSE"))
            cmd_close();
        else if (!strcmp(line, "APPEND"))
            cmd_append();
        else if (!strncmp(line, "INSERT", 6))
            cmd_insert(line + 6);
        else if (!strncmp(line, "EDIT", 4))
            cmd_edit(line + 4);
        else if (!strncmp(line, "GOTO", 4))
            cmd_goto(&db, line + 4);
        else if (!strncmp(line, "SKIP", 4))
            cmd_skip(line + 4);
        else if (!strncmp(line, "LOCATE", 6))
            cmd_locate(line + 6);
        else if (!strncmp(line, "CONT", 4))
            cmd_cont();
        else if (!strncmp(line, "SHOW", 4))
            cmd_show(line + 4);
        else if (!strncmp(line, "EXPORT", 6))
            cmd_export(line + 6);
        else if (!strncmp(line, "SORT", 4))
            cmd_sort(line + 4);
        else if (!strncmp(line, "DEL", 3))
            cmd_del(line + 3);
        else if (!strncmp(line, "UNDEL", 5))
            cmd_undel(line + 5);
        else if (!strcmp(line, "INFO"))
            cmd_info();
        else if (!strncmp(line, "HELP", 4))
            cmd_help(line + 4);
        else if (!strcmp(line, "CLS"))
            cmd_cls();
        else if (!strcmp(line, "DIR")) 
            cmd_dir();
        else if (!strcmp(line, "TEST")) 
            cmd_test();
        else if (!strcmp(line, "QUIT")) {
                if (db_opened && db.dirty) {
                    int ch = ask_yes_no(MSG_DB_DIRTY_WARNING);
                    printf("\r\n");
                    if (ch == DLG_YES)
                        break;
                }
                else
                    break;
            }
        else if (*line)
            printf(MSG_UNKNOWN_COMMAND);
        else
            printf("\r\n");
    }

   

    // Programm ende

    if (db_opened) {
        db_close(&db);
    }
    // Bildschirm löschen, Schaltet den Cursor wieder ein
    gp_clearscreen();
    return 0;
}



