/* CREATE.C */

/* CREATE Funktion zur Definition und Anlage einer neuen Datenbank*/

#include "create.h"
#include "messages.h"





// Parse für CREATE Erfassung der Feldstruktur
int parse_create_field(char *line, create_ctx_t *ctx)
{
    char *name, *type, *len, *dec;
    create_field_t *f;

    name = strtok(line, ",");
    type = strtok(NULL, ",");
    len  = strtok(NULL, ",");
    dec  = strtok(NULL, ",");

    if (!name || !type)
        return 0;

    if (dec == NULL)
        dec = "0";

    // Feldnamen prüfen    
    util_trim(name);
    util_str_toupper(name);

    if (strlen(name) == 0 || strlen(name) > 10)
        return 0;
    util_trim(type);
    if (strlen(type) != 1)
        return 0;

    // Feldtyp prüfen
    char ftype = type[0];

    if (ftype != 'C' && ftype != 'N' &&
        ftype != 'F' && ftype != 'L' &&
        ftype != 'D')
        return 0;

    // Feldlänge und Dezimalstellen
    int flen = 0;
    int fdec = 0;

    switch (ftype) {
    case 'C': // Character
        if (!len)
            return 0;
        flen = atoi(len);
        if (flen <= 0 || flen > 254)
            return 0;
        fdec = 0; // keine Dezimalstellen für C
        break;

    case 'N': // Numeric
    case 'F': // Float
        if (!len)
            return 0;
        flen = atoi(len);
        fdec = dec ? atoi(dec) : 0;
        if (flen <= 0 || flen > 254)
            return 0;
        if (fdec >= flen)
            return 0; // Dezimalstellen dürfen nicht > Länge sein
        break;

    case 'L': // Logical
        flen = 1;   // Länge 1 für Logical
        fdec = 0;   // keine Dezimalstellen für Logical
        break;

    case 'D': // Date
        flen = 8;   // Länge 8 für Datum (DD.MM.YYYY)
        fdec = 0;   // keine Dezimalstellen für Date
        break;

    default:
        return 0;
    }

    // Feld im Speicher anlegen    
    f = &ctx->fields[ctx->field_count];

    strncpy(f->name, name, sizeof(f->name) - 1);
    f->type = ftype;
    f->length = flen;
    f->decimals = fdec;

    ctx->record_len += flen;
    ctx->field_count++;

    // Felddefiniton erfolgreich
    return 1;
}


int create_save(const char *dbname, const create_field_t *fields, int field_count, uint16_t record_len)
{
    uint8_t *buf;
    size_t   buf_len;
    uint8_t *p;
    dbf_header_t *hdr;
    jdfcb_t fcb;

    time_t now;
    struct tm *lt;

    if (!dbname || !fields || field_count <= 0)
        return -1;

    /* Header + Felder + 0x0D + 0x1A + 0x00*/
    buf_len = sizeof(dbf_header_t)
            + field_count * sizeof(dbf_field_t)
            + 3;

    buf = malloc(buf_len);
    if (!buf)
        return -1;

    memset(buf, 0, buf_len);
    p = buf;

    /* ===== DBF HEADER ===== */
    hdr = (dbf_header_t *)p;

    now = _gettime();
    lt  = localtime(&now);

    hdr->version     = 0x03;          /* dBASE III */
    hdr->yy          = lt->tm_year;   /* seit 1900 */
    hdr->mm          = lt->tm_mon + 1;
    hdr->dd          = lt->tm_mday;
    hdr->num_records = to_le32(0);
    hdr->header_len  = to_le16(sizeof(dbf_header_t)
                          + field_count * sizeof(dbf_field_t)
                          + 1);
    hdr->record_len  = to_le16(record_len);
    p += sizeof(dbf_header_t);

    /* ===== FIELD DESCRIPTORS ===== */
    for (int i = 0; i < field_count; i++) {

        dbf_field_t df;
        memset(&df, 0, sizeof(df));

        strncpy(df.name, fields[i].name, sizeof(df.name));
        df.type     = fields[i].type;
        df.length   = fields[i].length;
        df.decimals = fields[i].decimals;

        memcpy(p, &df, sizeof(dbf_field_t));
        p += sizeof(dbf_field_t);
    }

    /* Header-Ende */
    *p++ = 0x0D;

    /* Dateiende */
    *p++ = 0x1A;

    /* JADOS Terminator */
    *p++ = 0x00;

    /* ===== Datei speichern ===== */
    jd_fillfcb(&fcb, (char *)dbname);

    if (jd_filesave(&fcb, buf, buf_len) != 0) {
        free(buf);
        return -1;
    }

    free(buf);
    return 0;
}


