/* DB.C  */
#include "cbase.h"
#include "db.h"
#include "util.h"
#include "locate.h"


/* --- Datenbank öffnen und komplett in den Speicher laden --- */

int db_open(db_t *db, const char *name)
{
    uint32_t off = 32;
    uint16_t rec_off = 1;   // Byte 0 = Lösch-Flag

    memset(db, 0, sizeof(*db));
    // Name in db->name einsetzen für INFO
    strncpy(db->name, name, DB_NAME_LEN - 1);
    db->name[DB_NAME_LEN - 1] = '\0';
    
    // Dateinamen bilden
    char fname[12];  /* 8.3 */

    make_db_filename(fname, sizeof(fname), name);


    /* FCB vorbereiten */
    if (jd_fillfcb(&db->fcb, fname) != 0)
        return -1;

    /* Datei öffnen */
    if (jd_open(&db->fcb) != 0)
        return -1;

    /* Puffergröße berechnen (1024 Byte pro Block) */
    db->buf_len = (size_t)db->fcb.length * 1024;

    if (db->buf_len == 0)
        return -1;

    /* Speicher allokieren */
    db->buf = (uint8_t *)malloc(db->buf_len);
    if (!db->buf)
        return -1;

    /* Datei komplett laden */
    if (jd_fileload(&db->fcb, db->buf) != 0)
        return -1;

    /* --- DBF Header einlesen --- */
    memcpy(&db->header, db->buf, sizeof(dbf_header_t));

    db->header.num_records = from_le32(db->header.num_records);
    db->header.header_len  = from_le16(db->header.header_len);
    db->header.record_len  = from_le16(db->header.record_len);

    /* --- Felddefinitionen einlesen --- */
    off = 32;
    db->field_count = 0;

    while (db->field_count < 16 && db->buf[off] != 0x0D) {

        dbf_field_t *f = &db->fields[db->field_count];

        memcpy(f, db->buf + off, sizeof(dbf_field_t));

        /* Feldname sicher terminieren */
        f->name[10] = 0;    // Dirtyflag setzen
        db->dirty = 1;


        /* dBASE-III: Offset seshow_separator(db_t *db)lbst berechnen */
        f->offset = rec_off;
        rec_off  += f->length;

        off += 32;
        db->field_count++;
    }

    // Dirtyflag zurücksetzen
    db->dirty = 0;
    // Für Navigation GOTO, SHOW, FIND LOCATE etc. aktueller Pointer auf Record
    db->current_rec = 0;
    db->eof = (db->header.num_records == 0);

    return 0;
}

/* --- Datenbank schließen --- */

void db_close(db_t *db)
{
    if (db->buf) {
        free(db->buf);
        db->buf = NULL;
    }

    jd_close(&db->fcb);
    db->name[0] = '\0';
}




// Anzeige von Datenbankinformationen zur aktuellen Datenbank
void db_info(const db_t *db)
{
    int i;
    printf("Database : %s\r\n", db->name[0] ? db->name : "<none>");
    printf("Records : %lu\r\n", (unsigned long)db->header.num_records);
    printf("Rec.len : %u\r\n",  db->header.record_len);
    printf("Header  : %u\r\n",  db->header.header_len);
    printf("Fields  : %d\r\n",  db->field_count);
    printf("DB-Dirty: %d\r\n",  db->dirty);
    printf("\r\n");
    printf("Memory  : %lu bytes\r\n", (unsigned long)db->buf_len);
    printf("Sectors : %lu\r\n", (unsigned long)(db->buf_len / 1024));
    printf("\r\n");

    printf("Nr Name        T Len Dec Off\r\n");
    printf("--------------------------------\r\n");

    for (i = 0; i < db->field_count; i++) {
        const dbf_field_t *f = &db->fields[i];

        printf("%2d %-11s %c %3u %3u %3u\r\n",
            i + 1,
            f->name,
            f->type,
            f->length,
            f->decimals,
            (unsigned)f->offset
        );
    }
}

// Logisches Löschen eines Datensatzes
int db_delete_record(db_t *db, uint32_t recno)
{
    uint8_t *r;

    /* Recordnummer prüfen (1-basiert!) */
    if (recno < 1 || recno > db->header.num_records)
        return -1;

    r = db->buf
        + db->header.header_len
        + (recno - 1) * db->header.record_len;

    /* schon gelöscht? */
    if (r[0] == '*')
        return 1;

    r[0] = '*';
    // Dirtyflag setzen
    db->dirty = 1;

    return 0;
}

int db_undel(db_t *db, uint32_t recno)
{
    uint8_t *r;

    if (recno >= db->header.num_records)
        return -1;

    r = db->buf
        + db->header.header_len
        + recno * db->header.record_len;

    if (r[0] != '*')
        return -1;   /* nicht gelöscht */

    r[0] = ' ';      /* Löschmarke entfernen */
    // Dirtyflag setzen
    db->dirty = 1;

    return 0;
}

// PACK
// PACK entfernt als gelöscht markierte Datensätze
// PACK speichert nicht
// Nach PACK erfolgt SAVE oder COPY
int db_pack(db_t *db)
{
    uint32_t src, dst = 0;

    for (src = 0; src < db->header.num_records; src++) {

        uint8_t *r = db->buf
            + db->header.header_len
            + src * db->header.record_len;

        if (r[0] == '*')
            continue;

        if (dst != src) {
            uint8_t *d = db->buf
                + db->header.header_len
                + dst * db->header.record_len;

            memcpy(d, r, db->header.record_len);
        }

        dst++;
    }

    db->header.num_records = dst;

    db->current_rec = 0;
    db->eof = (dst == 0);
    db->bof = (dst == 0);


    return dst;
}


/* Datenbank speichern*/
int db_save(db_t *db)
{
    dbf_header_t tmp;

    if (!db || !db->buf)
        return -1;

    /* Header in temporäre LE-Kopie */
    tmp = db->header;
    db_header_to_le(&tmp);

    /* Header zurück in Buffer schreiben (Sicherheit) */
    memcpy(db->buf, &tmp, sizeof(dbf_header_t));

    /* Datei schreiben */
    if (jd_filesave(&db->fcb, db->buf, db->buf_len))
        return -1;

    // Dirtyflag zurücksetzen
    db->dirty = 0;

    return 0;
}



int db_copy(db_t *db, const char *fname)
{
    jdfcb_t newfcb;
    dbf_header_t tmp;

    if (!db || !db->buf || !fname)
        return -1;

    /* Header LE-konvertiert in Buffer schreiben */
    tmp = db->header;
    db_header_to_le(&tmp);
    memcpy(db->buf, &tmp, sizeof(dbf_header_t));

    jd_fillfcb(&newfcb, (char *)fname);

    if (jd_filesave(&newfcb, db->buf, db->buf_len))
        return -1;

    return 0;

}


// Navigation in Datenbank, aktuellen Recordpointer setzen
int db_goto_record(db_t *db, uint32_t recno)
{
    if (db->header.num_records == 0) {
        db->eof = 1;
        return -1;
    }

    if (recno >= db->header.num_records) {
        db->eof = 1;
        return -1;
    }

    db->current_rec = recno;
    db->eof = 0;
    return 0;
}


// Navigation in Datenbank, aktuellen Recordpointer setzen TOP
int db_goto_top(db_t *db)
{
    if (db->header.num_records == 0) {
        db->eof = 1;
        return -1;
    }

    db->current_rec = 0;
    db->eof = 0;
    return 0;
}

// Navigation in Datenbank, aktuellen Recordpointer setzen BOTTOM
int db_goto_bottom(db_t *db)
{
    if (db->header.num_records == 0) {
        db->eof = 1;
        return -1;
    }

    db->current_rec = db->header.num_records - 1;
    db->eof = 0;
    return 0;
}


void db_skip(db_t *db, int32_t step)
{
    int dir = (step >= 0) ? 1 : -1;
    int remaining = abs(step);

    db->eof = 0;
    db->bof = 0;

    while (remaining > 0) {

        int32_t next = (int32_t)db->current_rec + dir;

        if (next < 0) {
            db->current_rec = 0;
            db->bof = 1;
            break;
        }

        if (next >= (int32_t)db->header.num_records) {
            db->current_rec = db->header.num_records - 1;
            db->eof = 1;
            break;
        }

        uint8_t *r = db->buf
            + db->header.header_len
            + next * db->header.record_len;

        db->current_rec = next;

        if (r[0] != '*')   /* nicht gelöscht */
            remaining--;
    }
}




