/* EXPORT.C*/

#include "cbase.h"
#include "db.h"
#include "util.h"
#include "locate.h"


// EXPORT Hilfsfunktionen




static uint8_t *csv_write_char_field(uint8_t *out,
                                     const uint8_t *data,
                                     unsigned len)
{
    unsigned i;

    *out++ = '"';

    for (i = 0; i < len; i++) {
        if (data[i] == '"') {
            *out++ = '"';   // CSV-Escape
            *out++ = '"';
        } else {
            *out++ = data[i];
        }
    }

    *out++ = '"';
    return out;
}


uint8_t *csv_write_num_field(uint8_t *out,
                             const uint8_t *data,
                             unsigned len)
{
    char buf[256];
    unsigned n;

    if (len >= sizeof(buf))
        len = sizeof(buf) - 1;

    n = util_copy_trimmed(buf, data, len);

    memcpy(out, buf, n);
    return out + n;
}




static uint8_t *csv_write_header(db_t *db, uint8_t *out)
{
    for (int i = 0; i < db->field_count; i++) {
        const char *name = db->fields[i].name;

        while (*name)
            *out++ = *name++;

        if (i < db->field_count - 1)
            *out++ = ';';
    }

    *out++ = '\r';
    *out++ = '\n';

    return out;
}

static uint8_t *csv_write_record(db_t *db,
                                 uint32_t recno,
                                 uint8_t *out)
{
    uint8_t *r = db->buf
        + db->header.header_len
        + recno * db->header.record_len;

for (int i = 0; i < db->field_count; i++) {

    const dbf_field_t *f = &db->fields[i];
    const uint8_t *data = r + f->offset;

    switch (f->type) {

        case 'C':   /* Character */
            out = csv_write_char_field(out, data, f->length);
            break;

        case 'N':   /* Numeric */
        case 'F':   /* Float */
        case 'L':   /* Logical */
        case 'D':   /* Date */
            out = csv_write_num_field(out, data, f->length);
            break;

        default:
            /* Fallback: roh, ungetrimmt */
            memcpy(out, data, f->length);
            out += f->length;
            break;
    }

    if (i < db->field_count - 1)
        *out++ = ';';
}

    *out++ = '\r';
    *out++ = '\n';

    return out;
}


// EXPORT
int db_export(db_t *db,
              const char *filename,
              db_export_mode_t mode,
              const locate_ctx_t *ctx)
{
    uint8_t *buf;
    uint8_t *out;
    size_t buf_len;
    jdfcb_t fcb;
    uint32_t recno;

    if (!db || !filename)
        return -1;

    /* grobe Abschätzung:
       Header + Records * (Recordlen * 2) */
    buf_len = 1024 + db->header.num_records * db->header.record_len * 2;

    buf = malloc(buf_len);
    if (!buf)
        return -1;

    out = buf;

    /* CSV-Header */
    out = csv_write_header(db, out);

    /* Datensätze */
    for (recno = 0; recno < db->header.num_records; recno++) {

        uint8_t *r = db->buf
            + db->header.header_len
            + recno * db->header.record_len;

        /* gelöscht überspringen */
        if (r[0] == '*')
            continue;

        if (mode == DB_EXPORT_CURRENT &&
            recno != db->current_rec)
            continue;

        if (mode == DB_EXPORT_LOC &&
            !db_record_matches_locate(db, recno, ctx))
            continue;

        out = csv_write_record(db, recno, out);
    }

    /* JADOS Terminator */
    *out++ = 0x00;

    /* Datei schreiben */
    jd_fillfcb(&fcb, filename);

    if (jd_filesave(&fcb, buf, out - buf) != 0) {
        free(buf);
        return -1;
    }

    free(buf);
    return 0;
}

