/* LOCATE.C */

#include "cbase.h"
#include "util.h"
#include "locate.h"

/*LOCATE Grundfunktion*/

int db_locate(db_t *db, const locate_ctx_t *ctx, uint32_t start_rec)
{
    uint32_t rec;

    if (!db || !ctx || !ctx->active)
        return -1;

    for (rec = start_rec; rec < db->header.num_records; rec++) {

        if (db_record_matches_locate(db, rec, ctx)) {
            db->current_rec = rec;
            db->eof = 0;
            return 0;
        }
    }

    db->eof = 1;
    return -1;
}


// COPY LOC kopiert LOCATE Selektion in neue Datenbank
int db_copy_loc(db_t *src, const char *dest_name, const locate_ctx_t *ctx)
{
    uint8_t *dst_buf;
    size_t dst_buf_len;
    uint32_t recno;
    uint32_t out_rec = 0;
    jdfcb_t fcb;
    dbf_header_t *hdr;
    uint8_t *dst_rec_ptr;
    uint8_t *src_rec_ptr;

    if (!src || !ctx || !ctx->active)
        return -1;

    /* Zielpuffer: gleiche Größe wie Quelle */
    dst_buf_len = src->buf_len;

    dst_buf = malloc(dst_buf_len);
    if (!dst_buf)
        return -1;

    memset(dst_buf, 0, dst_buf_len);

    /* Header kopieren */
    memcpy(dst_buf, src->buf, src->header.header_len);

    hdr = (dbf_header_t *)dst_buf;
    hdr->num_records = 0;  /* neu zählen */

    /* Ziel-Record-Pointer */
    dst_rec_ptr = dst_buf + src->header.header_len;

    /* alle Records prüfen */
    for (recno = 0; recno < src->header.num_records; recno++) {

        if (!db_record_matches_locate(src, recno, ctx))
            continue;

        src_rec_ptr = src->buf
            + src->header.header_len
            + recno * src->header.record_len;

        memcpy(dst_rec_ptr, src_rec_ptr, src->header.record_len);

        dst_rec_ptr += src->header.record_len;
        out_rec++;
    }

    dst_buf[src->header.header_len +
        out_rec * src->header.record_len] = 0x1A;

    /* Header finalisieren */
    hdr->num_records = to_le32(out_rec);


    /* Datei speichern */
    jd_fillfcb(&fcb, (char *)dest_name);

    if (jd_filesave(&fcb, dst_buf,
                    src->header.header_len +
                    out_rec * src->header.record_len + 1) != 0) {

        free(dst_buf);
        return -1;
    }

    free(dst_buf);
    return 0;
}



/*LOCATE */

// Für LOCATE Vergleich

int field_prefix_match(const uint8_t *field, unsigned field_len, const char *pattern)
{
    unsigned i = 0;

    while (i < field_len && pattern[i]) {

        char fc = field[i];
        char pc = pattern[i];

        /* Feldende (Blank) */
        if (fc == ' ')
            return 0;

        /* case-insensitive */
        if (fc >= 'a' && fc <= 'z')
            fc -= 32;
        if (pc >= 'a' && pc <= 'z')
            pc -= 32;

        if (fc != pc)
            return 0;

        i++;
    }

    /* Pattern komplett verbraucht → Treffer */
    return pattern[i] == '\0';
}




/// Hilfsroutine für LOCATE in cmd_locate
int find_field_index(db_t *db, const char *name)
{
    char tmp[12];
    int i;

    /* Feldname uppercase + trim */
    for (i = 0; i < 11 && name[i]; i++) {
        char c = name[i];
        if (c >= 'a' && c <= 'z')
            c -= 32;
        tmp[i] = c;
    }
    tmp[i] = '\0';

    for (i = 0; i < db->field_count; i++) {
        if (!strncmp(db->fields[i].name, tmp, 11))
            return i;
    }

    return -1;
}



// Auswerten des Vergleichsoperators in NUM Fields
locate_op_t parse_num_operator(const char **p)
{
    if (strncmp(*p, ">=", 2) == 0) {
        *p += 2;
        return LOC_OP_GE;
    }
    if (strncmp(*p, "<=", 2) == 0) {
        *p += 2;
        return LOC_OP_LE;
    }
    if (strncmp(*p, "!=", 2) == 0) {
        *p += 2;
        return LOC_OP_NE;
    }
    if (**p == '>') {
        (*p)++;
        return LOC_OP_GT;
    }
    if (**p == '<') {
        (*p)++;
        return LOC_OP_LT;
    }
    if (**p == '=') {
        (*p)++;
        return LOC_OP_EQ;
    }

    /* kein Operator → implizit = */
    return LOC_OP_EQ;
}






// Auswertung Datensätze für LOCATE
int db_record_matches_locate(db_t *db, uint32_t recno, const locate_ctx_t *ctx)
{
    uint8_t *r;

    if (!ctx || !ctx->active)
        return 0;

    r = db->buf
        + db->header.header_len
        + recno * db->header.record_len;

    if (r[0] == '*')
        return 0;

    /* Character-Feld */
    if (ctx->type == LOC_TYPE_CHAR) {

        char buf[256];
        unsigned len = db->fields[ctx->field_index].length;

        if (len >= sizeof(buf))
            len = sizeof(buf) - 1;

        memcpy(buf,
            r + db->fields[ctx->field_index].offset,
            len);
        buf[len] = 0;

        util_str_toupper(buf);

        return strstr(buf, ctx->str_value) != NULL;
    }
    /* Numeric-Feld */
    else if (ctx->type == LOC_TYPE_NUM) {

        char buf[64];
        unsigned len = db->fields[ctx->field_index].length;

        if (len >= sizeof(buf))
            len = sizeof(buf) - 1;

        memcpy(buf,
            r + db->fields[ctx->field_index].offset,
            len);
        buf[len] = '\0';

        double value = atof(buf);

        switch (ctx->op) {
        case LOC_OP_EQ:
            return value == ctx->num_value;
        case LOC_OP_NE:
            return value != ctx->num_value;
        case LOC_OP_LT:
            return value < ctx->num_value;
        case LOC_OP_LE:
            return value <= ctx->num_value;
        case LOC_OP_GT:
            return value > ctx->num_value;
        case LOC_OP_GE:
            return value >= ctx->num_value;
        default:
            return 0;
        }
    }

    return 0;
}


