/* RECORD.C Programme für APPEND, INSERT, EDIT*/


#include <stdio.h>
#include <string.h>
#include "record.h"
#include "messages.h"
#include "cbase.h"
#include "db.h"
#include "util.h"


// TEST für EDIT Bildschirmsteuerung
void test_screen_input(void)
{
    uint8_t x = 1, y = 1;
    int ch;
    //char buf[32];

    gp_clearscreen();
    gp_cursor_on();

    SetCursorxy(1,1);
    printf("GDP SCREEN TEST");
    SetCursorxy(1,3);
    printf("Use cursor keys, ESC to exit");

    SetCursorxy(1,10);
    iprintf("Hallo Welt!");
    SetCursorxy(x, y);

    while (1) {

        ch = gp_ci();

        if (ch == KEY_ESC)
            break;

        switch (ch) {

        case KEY_UP:
            if (y > 1) y--;
            break;

        case KEY_DOWN:
            if (y < 23) y++;
            break;

        case KEY_LEFT:
            if (x > 1) x--;
            break;

        case KEY_RIGHT:
            if (x < 79) x++;
            break;

        case KEY_RETURN:
            SetCursorxy(1,22);
            printf("RETURN pressed        ");
            x=1;
            if (y < 23) y++;
            break;
        
        case KEY_BACKSPACE:
        case KEY_BACKSPACE2:
            if (x > 1) x--;
                SetCursorxy(x,y);
                ch = 32;
                gp_co(ch);
            break;    

        default:
            if (ch >= 32 && ch <= 126) {
                SetCursorxy(x,y);
                gp_co(ch);
                if (x < 79) x++;
            }
            break;
        }

        SetCursorxy(x, y);

    }

    gp_cursor_off();
    gp_clearscreen();
}



// APPEND
int db_append_record(db_t *db)
{
    uint32_t new_rec;
    uint8_t *rec;
    int rc;

    if (!db || !db->buf)
        return APPEND_ABORT;

    new_rec = db->header.num_records;

    /* Platz prüfen */
    if ((db->header.header_len +
         (new_rec + 1) * db->header.record_len + 1) > db->buf_len) {

        printf(MSG_DB_NO_SPACE);
        return APPEND_ABORT;
    }

    /* Pointer auf neuen Record */
    rec = db->buf
        + db->header.header_len
        + new_rec * db->header.record_len;

    /* Initialisieren */
    memset(rec, ' ', db->header.record_len);
    rec[0] = ' ';

    /* <<< WICHTIG: Record temporär aktivieren */
    db->header.num_records++;

    rc = db_edit_record(db, new_rec);

    if (rc != EDIT_DONE) {
        /* Rollback */
        db->header.num_records--;
        return APPEND_SKIP;
    }

    /* EOF neu setzen */
    uint32_t eof_pos =
        db->header.header_len +
        db->header.num_records * db->header.record_len;

    db->buf[eof_pos] = 0x1A;

    return APPEND_DONE;
}


// Eingabe funktion
int append_read_field(
    char *buf,
    int maxlen,
    int (*valid_char)(int ch)
)
{
    int len = 0;
    int ch;

    gp_cursor_on();
    buf[0] = 0;

    while (1) {

        ch = gp_ci();

        /* ESC = Abbruch */
        if (ch == KEY_ESC) {
            buf[0] = 0;
            gp_cursor_off();
            return -1;
        }

        /* RETURN = fertig */
        if (ch == KEY_RETURN) {
            buf[len] = 0;
            iprintf("\r\n");
            gp_cursor_off();
            return len;
        }

        /* BACKSPACE */
        if (ch == 0x08 || ch == 0x7F) {
            if (len > 0) {
                len--;
                buf[len] = 0;
                iprintf("\b \b");
            }
            continue;
        }

        /* ungültiges Zeichen */
        if (valid_char && !valid_char(ch))
            continue;

        /* Feld voll */
        if (len >= maxlen) {
            //gp_beep();      /* optional */ TON Funktion? prüfen, gute Idee
            continue;
        }

        buf[len++] = (char)ch;
        iprintf("%c", ch);
    }
}



// INSERT
int db_insert(db_t *db, uint32_t insert_pos)
{
    uint8_t *new_buf;
    size_t new_len;


    if (insert_pos > db->header.num_records)
        return -1;

    new_len = db->header.header_len +
              (db->header.num_records + 1) * db->header.record_len + 1;

    new_buf = malloc(new_len);
    if (!new_buf)
        return -1;

    /* Header kopieren */
    memcpy(new_buf, db->buf, db->header.header_len);

    uint8_t *dst = new_buf + db->header.header_len;
    uint8_t *src = db->buf + db->header.header_len;

    /* Records vor insert_pos */
    memcpy(dst,
           src,
           insert_pos * db->header.record_len);

    dst += insert_pos * db->header.record_len;

    /* Neuer Record (leer, nicht gelöscht) */
    memset(dst, ' ', db->header.record_len);
    dst[0] = ' ';   // Löschflag

    dst += db->header.record_len;

    /* Restliche Records */
    memcpy(dst,
           src + insert_pos * db->header.record_len,
           (db->header.num_records - insert_pos) * db->header.record_len);

    /* EOF */
    new_buf[new_len - 1] = 0x1A;

    /* Header aktualisieren */
    dbf_header_t *hdr = (dbf_header_t *)new_buf;
    hdr->num_records = to_le32(db->header.num_records + 1);

    free(db->buf);
    db->buf = new_buf;
    db->buf_len = new_len;
    db->header.num_records++;

    return 0;
}


// EDIT
int db_edit_record(db_t *db, uint32_t recno)
{
    if (!db || recno >= db->header.num_records)
        return EDIT_ABORT;

    uint8_t *rec =
        db->buf +
        db->header.header_len +
        recno * db->header.record_len;

    gp_clearscreen();
    gp_cursor_on();

    SetCursorxy(1,1);
    printf("EDIT RECORD #%05lu", (unsigned long)(recno + 1));

    /* Felder zeichnen */
    for (int i = 0; i < db->field_count; i++) {
        edit_draw_field(&db->fields[i], rec, 3 + i);
    }

    int i = 0;

    while (i >= 0 && i < db->field_count) {

        int rc = edit_input_field(
            &db->fields[i],
            rec,
            3 + i
        );

        if (rc == FIELD_EXIT) {
            gp_cursor_off();
            return EDIT_ABORT;
        }

        if (rc == FIELD_NEXT) {
            if (i < db->field_count - 1)
                i++;
            else
                break;
        }

        if (rc == FIELD_PREV) {
            if (i > 0)
                i--;
        }
    }

    /* Speichern? */
    clear_status_line();
    SetCursorxy(1, 23);
    int ch = ask_yes_no(MSG_EDIT_SAVE_RECORD);

    gp_cursor_off();

    if (ch != DLG_YES)
        return EDIT_ABORT;

    db->dirty = 1;
    return EDIT_DONE;
}


int db_input_field(db_t *db, const dbf_field_t *field, uint8_t *rec, input_mode_t mode)
{
    char in[64];
    char out[64];
    int len;

    printf("%-10s : ", field->name);

    switch (field->type) {

    case 'C':
        len = append_read_field(in, field->length, valid_char_c);
        if (len < 0) return APPEND_FIELD_ABORT;

        memset(out, ' ', field->length);
        memcpy(out, in, len);
        break;

    case 'N':
        len = append_read_field(in, sizeof(in) - 1, valid_char_n);
        if (len < 0) return APPEND_FIELD_ABORT;

        if (!format_numeric(in, out, field->length, field->decimals)) {
            printf("INVALID NUMBER\r\n");
            return APPEND_FIELD_RETRY;
        }
        break;

    case 'L':
        len = append_read_field(in, 1, valid_char_l);
        if (len < 0) return APPEND_FIELD_ABORT;

        out[0] = (util_toupper(in[0])=='T' ||
                  util_toupper(in[0])=='Y' ||
                  util_toupper(in[0])=='J') ? 'T' : 'F';
        break;

    case 'D':
        len = append_read_field(in, 10, valid_char_d);
        if (len < 0)
            return APPEND_FIELD_ABORT;

        /* leeres Datum zulassen */
        if (len == 0) {
            memset(out, ' ', field->length);
            break;
        }

        if (!parse_date_input(in, out)) {
            printf("INVALID DATE (TT.MM.JJJJ)\r\n");
            return APPEND_FIELD_RETRY;
        }
        break;

    default:
        return APPEND_FIELD_ABORT;
    }

    memcpy(rec + field->offset, out, field->length);
    return APPEND_FIELD_OK;
}




// Formatierung N Type
int format_numeric(const char *in,
                   char *out,
                   unsigned width,
                   unsigned decimals)
{
    double value;
    long ip;
    long frac;
    double scale = 1.0;
    char tmp[64];
    char *p = tmp;

    while (*in == ' ')
        in++;

    if (*in == 0)
        return 0;

    value = strtod(in, NULL);

    /* Skalierung */
    for (unsigned i = 0; i < decimals; i++)
        scale *= 10.0;

    value = value >= 0 ? value + 0.0000001 : value - 0.0000001;

    ip = (long)value;
    frac = (long)((value - ip) * scale);

    if (frac < 0)
        frac = -frac;

    /* Ganzzahl */
    p += sprintf(p, "%ld", ip);

    /* Dezimalteil */
    if (decimals > 0) {
        *p++ = '.';
        sprintf(p, "%0*ld", decimals, frac);
    }

    if (strlen(tmp) > width)
        return 0;

    memset(out, ' ', width);
    memcpy(out + (width - strlen(tmp)), tmp, strlen(tmp));

    return 1;
}


// DATUMs Funktionen
// Formatierung D Type
int format_date(const char *in, char *out)
{
    int d,m,y;

    if (sscanf(in, "%2d.%2d.%4d", &d,&m,&y) != 3)
        return 0;

    if (d<1||d>31||m<1||m>12||y<1900)
        return 0;

    snprintf(out, 9, "%04d%02d%02d", y,m,d);
    return 1;
}




// Eingabefilter C Type
int valid_char_c(int ch)
{
    return (ch >= 32 && ch <= 126);
}

// Eingabefilter N Type
int valid_char_n(int ch)
{
    if (ch >= '0' && ch <= '9') return 1;
    if (ch == '-' || ch == '.') return 1;
    return 0;
}


// Eingabefilter L Type
int valid_char_l(int ch)
{
    ch = util_toupper(ch);
    return ch=='T'||ch=='F'||ch=='J'||ch=='Y'||ch=='N';
}

// Eingabefilter D Type
int valid_char_d(int ch)
{
    return (ch >= '0' && ch <= '9') || ch == '.';
}



// Hilfsfunktion zur Anzeige des Eingabeprompts "Feldname    :"
void print_field_prompt(const char *name)
{
    printf("%-10s : ", name);
}


// EDIT Statuszeile löschen
void clear_status_line(void)
{
    SetCursorxy(1, 23);
    printf("%-79s", "");
}


void edit_draw_field(const dbf_field_t *f, const uint8_t *rec, uint8_t y)
{
    char buf[32];

    SetCursorxy(1, y);
    printf("%-10s : ", f->name);

    SetCursorxy(14, y);

    if (f->type == 'D') {
        format_date_display(rec, f->offset, buf);
        printf("%s", buf);
        return;
    }

    for (int i = 0; i < f->length; i++) {
        char c = rec[f->offset + i];
        if (c < 32) c = ' ';
        gp_co(c);
    }
}


// Datum in EDIT
// Schaltjahr erkennung
// static int is_leap_year(int y)
// {
//     return (y % 4 == 0 && y % 100 != 0) || (y % 400 == 0);
// }

// Mögliche Anzahl Tage in Monaten mit Schaltjahrberücksichtigung
// static int days_in_month(int y, int m)
// {
//     static const int d[12] = {
//         31,28,31,30,31,30,31,31,30,31,30,31
//     };

//     if (m == 2 && is_leap_year(y))
//         return 29;

//     return d[m - 1];
// }

// Datumskonvertierung und Parser
int parse_date_input(const char *in, char *out_yyyymmdd)
{
#if DATEFORMAT == DATEFMT_DDMMYYYY
    int d, m, y;
    if (sscanf(in, "%2d.%2d.%4d", &d, &m, &y) != 3)
        return 0;
#elif DATEFORMAT == DATEFMT_MMDDYYYY
    int d, m, y;
    if (sscanf(in, "%2d/%2d/%4d", &m, &d, &y) != 3)
        return 0;
#elif DATEFORMAT == DATEFMT_YYYYMMDD
    int d, m, y;
    if (sscanf(in, "%4d-%2d-%2d", &y, &m, &d) != 3)
        return 0;
#endif

    if (y < 1582 || y > 2300) return 0;
    if (m < 1 || m > 12) return 0;
    if (d < 1 || d > 31) return 0;

    sprintf(out_yyyymmdd, "%04d%02d%02d", y, m, d);
    return 1;
}


// EDIT Eingabezeile bearbeiten je nach Feld Typ
int edit_input_field(const dbf_field_t *f, uint8_t *rec, uint8_t y)
{
    char buf[11];        /* Anzeigeformat */
    int cur = 0;
    int ch;

    if (f->type == 'D') {

        char out8[9];    /* YYYYMMDD */
        int dirty = 0;

        /* bestehendes Datum anzeigen */
        format_date_display(rec, f->offset, buf);

        SetCursorxy(14, y);
        printf("%s", buf);

        /* Separator-Positionen */
        auto int is_sep_pos(int p)
        {
        #if DATEFORMAT == DATEFMT_DDMMYYYY
            return (p == 2 || p == 5);
        #elif DATEFORMAT == DATEFMT_MMDDYYYY
            return (p == 2 || p == 5);
        #elif DATEFORMAT == DATEFMT_YYYYMMDD
            return (p == 4 || p == 7);
        #endif
        }

        auto int next_pos(int p)
        {
            do { p++; } while (p < 10 && is_sep_pos(p));
            if (p > 9) p = 9;
            return p;
        }

        auto int prev_pos(int p)
        {
            do { p--; } while (p > 0 && is_sep_pos(p));
            if (p < 0) p = 0;
            return p;
        }

        /* Cursor auf erste Ziffer */
        cur = 0;
        SetCursorxy(14 + cur, y);

        while (1) {

            ch = gp_ci();

            switch (ch) {

                case KEY_ESC:
                    return FIELD_EXIT;

                case KEY_UP:
                    return FIELD_PREV;

                case KEY_DOWN:
                    return FIELD_NEXT;

                case KEY_LEFT:
                    cur = prev_pos(cur);
                    break;

                case KEY_RIGHT:
                    cur = next_pos(cur);
                    break;

                case KEY_RETURN:
                    {
                        /* leeres Feld zulassen */
                        int empty = 1;
                        for (int i = 0; i < 10; i++)
                            if (buf[i] >= '0' && buf[i] <= '9')
                                empty = 0;

                        if (empty) {
                            memset(rec + f->offset, ' ', 8);
                            return FIELD_NEXT;
                        }

                        if (!parse_date_input(buf, out8)) {
                            clear_status_line();
                            SetCursorxy(1, 23);
                            printf("INVALID DATE (%s)     ",
                        #if DATEFORMAT == DATEFMT_DDMMYYYY
                                                "TT.MM.JJJJ"
                        #elif DATEFORMAT == DATEFMT_MMDDYYYY
                                                "MM/DD/YYYY"
                        #elif DATEFORMAT == DATEFMT_YYYYMMDD
                                                "YYYY-MM-DD"
                        #endif
                            );
                            SetCursorxy(14 + cur, y);
                            break;
                        }

                        memcpy(rec + f->offset, out8, 8);
                        return FIELD_NEXT;
                    }

                default:
                    if (ch >= '0' && ch <= '9') {

                        if (!dirty) {
                        #if DATEFORMAT == DATEFMT_DDMMYYYY
                                                strcpy(buf, "  .  .    ");
                        #elif DATEFORMAT == DATEFMT_MMDDYYYY
                                                strcpy(buf, "  /  /    ");
                        #elif DATEFORMAT == DATEFMT_YYYYMMDD
                                                strcpy(buf, "    -  -  ");
                        #endif
                            dirty = 1;
                        }

                        buf[cur] = ch;

                        SetCursorxy(14, y);
                        printf("%s", buf);

                        cur = next_pos(cur);
                    }
                    break;
            }

            SetCursorxy(14 + cur, y);
        }

    }
    else
    {
        char in[256];
        char tmp[256];
        int first_key = 1;

        memset(buf, ' ', f->length);
        memcpy(buf, rec + f->offset, f->length);

        SetCursorxy(14, y);

        while (1) {

            ch = gp_ci();

            switch (ch) {

            case KEY_ESC:
                return FIELD_EXIT;

            case KEY_UP:
                memcpy(rec + f->offset, buf, f->length);
                return FIELD_PREV;

            case KEY_DOWN:
                memcpy(rec + f->offset, buf, f->length);
                return FIELD_NEXT;

            case KEY_RETURN: {

                if (f->type == 'N') {

                    int has_input = 0;
                    for (int i = 0; i < f->length; i++)
                        if (buf[i] != ' ')
                            has_input = 1;

                    if (has_input) {
                        memcpy(in, buf, f->length);
                        in[f->length] = 0;   /* <<< JETZT WIRKSAM */

                        if (!format_numeric(in, tmp,
                                            f->length,
                                            f->decimals)) {
                            clear_status_line();                    
                            SetCursorxy(1, 23);
                            printf("INVALID NUMBER        ");
                            SetCursorxy(14 + cur, y);
                            break;
                        }
                        memcpy(buf, tmp, f->length);
                    } else {
                        memcpy(buf, rec + f->offset, f->length);
                    }
                }

                memcpy(rec + f->offset, buf, f->length);
                return FIELD_NEXT;
            }

            case KEY_LEFT:
                if (cur > 0) cur--;
                break;

            case KEY_RIGHT:
                if (cur < f->length - 1) cur++;
                break;

            case KEY_BACKSPACE:
            case KEY_BACKSPACE2:
                if (cur > 0) {
                    cur--;
                    buf[cur] = ' ';
                    SetCursorxy(14 + cur, y);
                    gp_co(' ');
                }
                break;

            default:
                if (ch >= 32 && ch <= 126) {

                    /* ===== L-FELD ===== */
                    if (f->type == 'L') {

                        /* in Uppercase wandeln */
                        if (ch >= 'a' && ch <= 'z')
                            ch = ch - 'a' + 'A';

                        /* nur gültige Logical-Werte erlauben */
                        if (!strchr("TFJNY", ch))
                            break;

                        buf[0] = ch;
                        gp_co(ch);

                        memcpy(rec + f->offset, buf, 1);
                        return FIELD_NEXT;
                    }

                    /* ===== N-FELD ===== */
                    if (first_key && f->type == 'N') {
                        memset(buf, ' ', f->length);
                        cur = 0;
                        SetCursorxy(14, y);
                    }

                    first_key = 0;

                    if (cur < f->length) {
                        buf[cur++] = ch;
                        gp_co(ch);
                    }
                }
                break;
            }

            SetCursorxy(14 + cur, y);
        }
    }
}
