/* SHOW.C */
#include <stdio.h>
#include <stdint.h>

#include "cbase.h"
#include "show.h"
#include "util.h"
#include "locate.h"
#include "messages.h"

#define SHOW_PAGE_LINES 17


static void show_header(db_t *db)
{
    int j;

    /* Recordnummer + Marker */
    printf(" #   ");

    for (j = 0; j < db->field_count; j++) {

        int width = db->fields[j].length;

        /* Datum wird immer 10-stellig angezeigt */
        if (db->fields[j].type == 'D')
            width = 10;

        /* Feldname linksbündig, aber hart auf width begrenzen */
        printf("%-*.*s ", width, width, db->fields[j].name);
    }

    printf("\n");
}



// Trennstrich zwischen Feldbezeichner und Datenbankende
// Trennstrich nach Datenausgabe vor Statuszeile
static void show_separator(db_t *db)
{
    int i, j;

    /* Recordnummer + Marker */
    printf("---  ");

    for (j = 0; j < db->field_count; j++) {

        int width = db->fields[j].length;

        if (db->fields[j].type == 'D')
            width = 10;   /* Anzeigeformat */

        for (i = 0; i < width; i++)
            putchar('-');

        putchar(' ');
    }
    putchar('\n');
}



/* --- Unterfunktion für db_list --- */

void print_header_field(const char *name, unsigned width)
{
    unsigned i = 0;

    while (i < width && name[i]) {
        putchar(name[i]);
        i++;
    }

    while (i < width) {
        putchar(' ');
        i++;
    }
}

void print_data_field(const dbf_field_t *f, const uint8_t *p)
{
    unsigned i;

    if (f->type == 'D') {
        char buf[11];
        format_date_for_show(p, buf);
        printf("%-10s", buf);
        return;
    }

    /* alle anderen Typen unverändert */
    for (i = 0; i < f->length; i++)
        putchar(p[i] ? p[i] : ' ');
}






static void show_status(db_t *db, uint32_t start_rec, uint32_t page_start, uint32_t printed)
{
    printf(
        "Records %lu-%lu of %lu\r\n",
        (unsigned long)(start_rec + page_start + 1),
        (unsigned long)(start_rec + printed),
        (unsigned long)db->header.num_records
    );
}

// Formatanpassung für Datumsfeld
void format_date_for_show(const uint8_t *p, char *out)
{
    /* leeres Datum */
    if (p[0] == ' ' || p[0] == 0) {
#if DATEFORMAT == DATEFMT_DDMMYYYY
        strcpy(out, "  .  .    ");
#elif DATEFORMAT == DATEFMT_MMDDYYYY
        strcpy(out, "  /  /    ");
#elif DATEFORMAT == DATEFMT_YYYYMMDD
        strcpy(out, "    -  -  ");
#endif
        return;
    }

#if DATEFORMAT == DATEFMT_DDMMYYYY
    /* YYYYMMDD → DD.MM.YYYY */
    out[0] = p[6];
    out[1] = p[7];
    out[2] = '.';
    out[3] = p[4];
    out[4] = p[5];
    out[5] = '.';
    out[6] = p[0];
    out[7] = p[1];
    out[8] = p[2];
    out[9] = p[3];
#elif DATEFORMAT == DATEFMT_MMDDYYYY
    /* YYYYMMDD → MM/DD/YYYY */
    out[0] = p[4];
    out[1] = p[5];
    out[2] = '/';
    out[3] = p[6];
    out[4] = p[7];
    out[5] = '/';
    out[6] = p[0];
    out[7] = p[1];
    out[8] = p[2];
    out[9] = p[3];
#elif DATEFORMAT == DATEFMT_YYYYMMDD
    /* YYYYMMDD → YYYY-MM-DD */
    out[0] = p[0];
    out[1] = p[1];
    out[2] = p[2];
    out[3] = p[3];
    out[4] = '-';
    out[5] = p[4];
    out[6] = p[5];
    out[7] = '-';
    out[8] = p[6];
    out[9] = p[7];
#endif

    out[10] = 0;
}


void cbase_show(db_t *db, uint32_t start_rec, uint32_t max_count, db_show_mode_t mode)
{
    uint32_t recno = start_rec;
    uint32_t page_start = 0;

    uint32_t shown;

    uint32_t printed = 0;

    while (recno < db->header.num_records) {

        page_start = printed;
        show_header(db);
        show_separator(db);

        shown = 0;
        while (shown < SHOW_PAGE_LINES &&
               recno < db->header.num_records) {

            uint8_t *r = db->buf
                + db->header.header_len
                + recno * db->header.record_len;

            /* Löschmarkierung prüfen */
            if (mode == DB_SHOW_NORMAL && r[0] == '*')
            {
                recno++;
                continue;
            }

            if (mode == DB_SHOW_DELETED && r[0] != '*')
            {
                recno++;
                continue;
            }

            // Recordnumber ausgeben
            //printf("%3lu", (unsigned long)recno);
            printf("%3lu ", (unsigned long)(recno + 1));

            // Datesatzzeiger für aktiven Satz ausgeben
            if (recno == db->current_rec)
                printf(">");
            else
                printf(" ");

            // Über Datenfelder iterieren und ausgeben
            for (int j = 0; j < db->field_count; j++) {
                print_data_field(
                    &db->fields[j],
                    r + db->fields[j].offset
                );

                putchar(' ');
            }

            printf("\r\n");

            // bereits angzeigte Datensätze
            printed++;

            // Ausgabe auf maximal gewünschte Datensätze SHOW 5 5 < zweiter Parameter begrenzen
            if (max_count && printed >= max_count) {
                show_separator(db);
                show_status(db, start_rec, page_start, printed);
                return;
            }

            // Gezeigte Datensätze für Paging
            shown++;
            recno++;
        }
        // Statuszeile anzeigen nach Seite
        if (printed > page_start) {
            show_separator(db);
            show_status(db, start_rec, page_start, printed);
        }
        // Datenbankende erreicht, Ende
        if (recno >= db->header.num_records)
            break;

        // Seitenwechsel mit Return oder ESC
        iprintf(MSG_SHOW_PAGE);

        if (wait_continue_or_quit() != DLG_CONTINUE) {
            iprintf("\r\n");
            break;
        }

        iprintf("\r\n");

    }

}
// Zugriff auf Struct, definiert in cbase.c
extern locate_ctx_t locate_ctx;

// SHOW LOC - LOCATE auswerten und anzeigen
void cbase_show_loc(db_t *db, locate_ctx_t *locate_ctx)
{
    uint32_t recno;
    uint32_t shown = 0;

    show_header(db);
    show_separator(db);

    for (recno = 0; recno < db->header.num_records; recno++) {

        if (!db_record_matches_locate(db, recno, locate_ctx))
            continue;

        uint8_t *r = db->buf
            + db->header.header_len
            + recno * db->header.record_len;

        /* Recordnummer (1-basiert) */
        printf("%3lu ", (unsigned long)(recno + 1));

        /* Aktuellen Satz markieren */
        if (recno == db->current_rec)
            printf(">");
        else
            printf(" ");

        for (int j = 0; j < db->field_count; j++) {
            print_data_field(
                &db->fields[j],
                r + db->fields[j].offset
            );
            putchar(' ');
        }

        printf("\r\n");
        shown++;
    }

    show_separator(db);

    if (shown == 0) {
        // Keine Datensätze gefunden
        printf(MSG_NO_RECORDS);
    }
    else {
        printf("%lu Records(S)\r\n", (unsigned long)shown);
    }

}

