// SORT.C
#include "cbase.h"
#include "db.h"
#include "util.h"
#include "sort.h"


static sort_ctx_t sort_ctx;


#include "cbase.h"
#include "db.h"
#include "util.h"
#include "sort.h"

static sort_ctx_t sort_ctx;

/* -------------------------------------------------------
   Comparator für qsort – Multi-Key
   ------------------------------------------------------- */
int db_sort_cmp(const void *a, const void *b)
{
    uint32_t ra = *(const uint32_t *)a;
    uint32_t rb = *(const uint32_t *)b;

    db_t *db = sort_ctx.db;

    for (int k = 0; k < sort_ctx.key_count; k++) {

        const sort_key_t *key = &sort_ctx.keys[k];
        const dbf_field_t *f  = &db->fields[key->field_index];

        uint8_t *pa = db->buf
            + db->header.header_len
            + ra * db->header.record_len
            + f->offset;

        uint8_t *pb = db->buf
            + db->header.header_len
            + rb * db->header.record_len
            + f->offset;

        int result = 0;

        /* -------------------------------
           Character
           ------------------------------- */
        if (f->type == 'C') {

            char sa[256], sb[256];
            unsigned len = f->length;

            if (len >= sizeof(sa))
                len = sizeof(sa) - 1;

            memcpy(sa, pa, len);
            memcpy(sb, pb, len);
            sa[len] = sb[len] = 0;

            util_str_toupper(sa);
            util_str_toupper(sb);

            result = strcmp(sa, sb);
        }

        /* -------------------------------
           Numeric / Float
           ------------------------------- */
        else if (f->type == 'N' || f->type == 'F') {

            char sa[64], sb[64];
            double da, dbv;

            memcpy(sa, pa, f->length);
            memcpy(sb, pb, f->length);
            sa[f->length] = sb[f->length] = 0;

            da  = atof(sa);
            dbv = atof(sb);

            if (da < dbv) result = -1;
            else if (da > dbv) result = 1;
            else result = 0;
        }

        /* -------------------------------
           Logical (F < T)
           ------------------------------- */
        else if (f->type == 'L') {

            char la = util_toupper(pa[0]);
            char lb = util_toupper(pb[0]);

            if (la < lb) result = -1;
            else if (la > lb) result = 1;
            else result = 0;
        }

        /* -------------------------------
           Date (YYYYMMDD)
           ------------------------------- */
        else if (f->type == 'D') {

            result = memcmp(pa, pb, 8);
        }

        /* Ergebnis auswerten */
        if (result != 0) {
            return key->descending ? -result : result;
        }
        /* sonst → nächster Sort-Key */
    }

    return 0;
}

/* -------------------------------------------------------
   Sort vorbereiten (Indexliste erzeugen)
   ------------------------------------------------------- */
int db_sort_prepare(db_t *db, const sort_key_t *keys, int key_count, uint32_t **out_index, uint32_t *out_count)
{
    uint32_t *idx;
    uint32_t i, count = 0;

    if (!db || !keys || key_count <= 0 || !out_index || !out_count)
        return -1;

    idx = malloc(sizeof(uint32_t) * db->header.num_records);
    if (!idx)
        return -1;

    /* nur nicht-gelöschte Records */
    for (i = 0; i < db->header.num_records; i++) {

        uint8_t *r = db->buf
            + db->header.header_len
            + i * db->header.record_len;

        if (r[0] != '*')
            idx[count++] = i;
    }

    sort_ctx.db        = db;
    sort_ctx.key_count = key_count;
    memcpy(sort_ctx.keys, keys, sizeof(sort_key_t) * key_count);

    qsort(idx, count, sizeof(uint32_t), db_sort_cmp);

    *out_index = idx;
    *out_count = count;
    return 0;
}


int db_sort_to_file(db_t *src, const char *dest_name, const uint32_t *idx, uint32_t count)
{
    uint8_t *dst_buf;
    size_t dst_size;
    jdfcb_t fcb;
    dbf_header_t *hdr;
    uint8_t *dst_rec;
    uint8_t *src_rec;
    uint32_t i;

    if (!src || !dest_name || !idx || count == 0)
        return -1;

    /* Zielpuffer:
       Header + (count Records) + EOF */
    dst_size = src->header.header_len
             + count * src->header.record_len
             + 1;

    dst_buf = malloc(dst_size);
    if (!dst_buf)
        return -1;

    memset(dst_buf, 0, dst_size);

    /* Header + Felddefinitionen kopieren */
    memcpy(dst_buf, src->buf, src->header.header_len);

    hdr = (dbf_header_t *)dst_buf;
    hdr->num_records = to_le32(count);

    dst_rec = dst_buf + src->header.header_len;

    /* Datensätze in sortierter Reihenfolge kopieren */
    for (i = 0; i < count; i++) {

        src_rec = src->buf
            + src->header.header_len
            + idx[i] * src->header.record_len;

        memcpy(dst_rec, src_rec, src->header.record_len);
        dst_rec += src->header.record_len;
    }

    /* EOF-Marker */
    *dst_rec = 0x1A;

    /* Datei speichern */
    jd_fillfcb(&fcb, (char *)dest_name);

    if (jd_filesave(&fcb, dst_buf, dst_size) != 0) {
        free(dst_buf);
        return -1;
    }

    free(dst_buf);
    return 0;
}
