/* UTIL.C */
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <ctype.h>
#include "../../nkc_common/nkc/nkc.h"

#include "cbase.h"
#include "db.h"
#include "util.h"


// Für den DIR Befehl
#define DIR_BUF_SIZE 4096

static char history[HIST_MAX][128];
static int  hist_count = 0;
static int  hist_pos   = -1;



/* --- Endian-Hilfsfunktionen (68000 = Big Endian, DBF = Little Endian) --- */

uint16_t from_le16(uint16_t v)
{
    return (uint16_t)((v >> 8) | (v << 8));
}

uint32_t from_le32(uint32_t v)
{
    return ((v >> 24) & 0x000000FFUL) |
           ((v >>  8) & 0x0000FF00UL) |
           ((v <<  8) & 0x00FF0000UL) |
           ((v << 24) & 0xFF000000UL);
}



uint16_t to_le16(uint16_t v)
{
    return (uint16_t)((v >> 8) | (v << 8));
}

uint32_t to_le32(uint32_t v)
{
    return ((v >> 24) & 0x000000FFUL) |
           ((v >>  8) & 0x0000FF00UL) |
           ((v <<  8) & 0x00FF0000UL) |
           ((v << 24) & 0xFF000000UL);
}


void db_header_to_le(dbf_header_t *h)
{
    h->num_records = to_le32(h->num_records);
    h->header_len  = to_le16(h->header_len);
    h->record_len  = to_le16(h->record_len);
}



// Funktio für Umwandlung in Uppercase
// diese Funktion arbeitet ohne Rückgabewert und 
// modifiziert den String im Speicher direkt (Inplace)
void util_str_toupper(char *s)
{
    while (*s) {
        if (*s >= 'a' && *s <= 'z')
            *s -= ('a' - 'A');
        s++;
    }
}

// Zeichenweise Umwandlung in Uppercase mit Rückgabe des Zeichen
char util_toupper(char c)
{
    if (c >= 'a' && c <= 'z')
        c -= ('a' - 'A');
    return c;
}


/* entfernt führende und folgende Leerzeichen */
void util_trim(char *s)
{
    char *end;

    /* leading */
    while (*s == ' ')
        s++;

    /* trailing */
    end = s;
    while (*end)
        end++;
    while (end > s && end[-1] == ' ')
        end--;
    *end = 0;
}

unsigned util_copy_trimmed(char *dst, const uint8_t *src, unsigned len)
{
    int i;

    /* rechts trimmen */
    for (i = len - 1; i >= 0; i--) {
        if (src[i] != ' ')
            break;
    }

    /* kopieren */
    for (int j = 0; j <= i; j++)
        dst[j] = src[j];

    return (i >= 0) ? (unsigned)(i + 1) : 0;
}


void util_print_field_trimmed(const uint8_t *p, unsigned len)
{
    char buf[256];
    unsigned n;

    if (len >= sizeof(buf))
        len = sizeof(buf) - 1;

    n = util_copy_trimmed(buf, p, len);
    buf[n] = 0;

    iprintf("%s", buf);
}


// DATUMS FORMAT Einstellung
void format_date_display(const uint8_t *rec, int offset, char *out)
{
    char y[5], m[3], d[3];

    memcpy(y, rec + offset + 0, 4);
    memcpy(m, rec + offset + 4, 2);
    memcpy(d, rec + offset + 6, 2);

    y[4] = m[2] = d[2] = 0;

#if DATEFORMAT == DATEFMT_DDMMYYYY
    sprintf(out, "%s.%s.%s", d, m, y);
#elif DATEFORMAT == DATEFMT_MMDDYYYY
    sprintf(out, "%s/%s/%s", m, d, y);
#elif DATEFORMAT == DATEFMT_YYYYMMDD
    sprintf(out, "%s-%s-%s", y, m, d);
#endif
}



int util_starts_with(const char *s, const char *prefix)
{
    while (*prefix) {
        if (*s++ != *prefix++)
            return 0;
    }
    return 1;
}



// Funktion zum Einlesen des aktuellen JADOS Verzeichnisses
void util_dir_drive(const char *pattern)
{
    static char buf[DIR_BUF_SIZE];
    uint8_t result;
    char *p;

    if (!pattern)
        pattern = "*.*";

    result = jd_directory(buf, pattern, 0u, 4u, sizeof(buf));

    if (result != 0) {
        printf("\nDIR error: 0x%02X\r\n", result);
        return;
    }

    printf("\nDirectory (%s)\r\n\r\n", pattern);

    /* JADOS liefert bereits Text → direkt ausgeben */
    p = buf;
    while (*p) {
        putchar(*p);
        p++;
    }
}


int is_valid_db_name(const char *name)
{
    int len = 0;

    if (!name || !*name)
        return 0;

    /* "." oder ".." verbieten */
    if ((name[0] == '.' && name[1] == '\0') ||
        (name[0] == '.' && name[1] == '.' && name[2] == '\0'))
        return 0;

    while (*name) {
        char c = *name++;

        /* nur A–Z, a–z, 0–9, _ erlauben */
        if (!((c >= 'A' && c <= 'Z') ||
              (c >= 'a' && c <= 'z') ||
              (c >= '0' && c <= '9') ||
              c == '_'))
            return 0;

        len++;
        if (len > 8)   /* JADOS 8.3 */
            return 0;
    }

    return 1;
}



/* Datenbanknamen bilden */
void make_db_filename(char *dst, size_t dstlen, const char *name)
{
    snprintf(dst, dstlen, "%s%s", name, DB_SUFFIX);
}


/* CSV Dateinamen bilden*/
void make_csv_filename(char *out, size_t len, const char *name)
{
    snprintf(out, len, "%s.CSV", name);
}



// Aktuelle datensatznummer ausgeben (1 - basiert für externe Zuordnung)
void print_current_record(db_t *db)
{
    printf("(record %lu)\r\n",
           (unsigned long)(db->current_rec + 1));
}







// Routinen für Eingabe
int read_key(void)
{
    int ch;
    gp_cursor_on();
    ch = gp_ci();
    gp_cursor_off();

    return ch;
}


int ask_yes_no(const char *prompt)
{
    int ch;

    iprintf("%s (J/N) ", prompt);

    while (1) {

        ch = read_key();

        if (ch == KEY_ESC)
            return DLG_ABORT;

        if (ch == KEY_YES_UPPER || ch == KEY_YES_LOWER)
            return DLG_YES;

        if (ch == KEY_NO_UPPER || ch == KEY_NO_LOWER)
            return DLG_NO;
    }
}


int wait_continue_or_quit(void)
{
    int ch = read_key();

    if (ch == KEY_ESC)
        return DLG_ABORT;

    if (ch == KEY_RETURN || ch == KEY_NEWLINE)
        return DLG_CONTINUE;

    return DLG_CONTINUE;
}

// Befehl von der Eingabe im History Speicher ablegen
void history_add(const char *line)
{
    if (!line || !*line)
        return;

    /* keine doppelten Direkt-Wiederholungen */
    if (hist_count > 0 &&
        strcmp(history[(hist_count - 1) % HIST_MAX], line) == 0)
        return;

    strncpy(history[hist_count % HIST_MAX], line, sizeof(history[0]) - 1);
    history[hist_count % HIST_MAX][sizeof(history[0]) - 1] = '\0';

    hist_count++;
    hist_pos = hist_count;   /* Reset Position */
}

// Aktuelle Engabezeile löschen
static void clear_line(int len)
{
    while (len--) {
        iprintf("\b \b");
    }
}



// Eingabefunktion für Befehlseingabe
int read_line_ci(char *buf, int maxlen)
{
    int len = 0;
    int ch;

    gp_cursor_on();

    while (1) {
        ch = gp_ci();

        /* ESC = Abbruch */
        if (ch == KEY_ESC) {
            buf[0] = '\0';
            gp_cursor_off();
            return -1;
        }

        /* RETURN = fertig */
        if (ch == KEY_RETURN) {
            buf[len] = '\0';
            iprintf("\r\n");
            gp_cursor_off();
            return len;
        }
        
        /* Pfeil hoch = vorheriges Kommando */
        if (ch == KEY_UP) {
            if (hist_count > 0 && hist_pos > 0) {
                clear_line(len);
                hist_pos--;
                strncpy(buf, history[hist_pos % HIST_MAX], maxlen - 1);
                buf[maxlen - 1] = '\0';
                len = strlen(buf);
                iprintf("%s", buf);
            }
            continue;
        }

        /* Pfeil runter = nächstes Kommando */
        if (ch == KEY_DOWN) {
            if (hist_pos < hist_count - 1) {
                clear_line(len);
                hist_pos++;
                strncpy(buf, history[hist_pos % HIST_MAX], maxlen - 1);
                buf[maxlen - 1] = '\0';
                len = strlen(buf);
                iprintf("%s", buf);
            } else {
                clear_line(len);
                len = 0;
                buf[0] = '\0';
                hist_pos = hist_count;
            }
            continue;
        }

        /* BACKSPACE */
        if (ch == 0x08 || ch == 0x7F) {
            if (len > 0) {
                len--;
                buf[len] = '\0';
                iprintf("\b \b");   /* Zeichen löschen */
            }
            continue;
        }

        /* druckbare Zeichen */
        if (ch >= 32 && ch <= 126) {
            if (len < maxlen - 1) {
                buf[len++] = (char)ch;
                iprintf("%c", ch);
            }
        }
    }
}
